'use client';
import { Upload } from 'lucide-react';
import { useInventory } from '../providers/inventory-provider';
import { useToast } from '@/hooks/use-toast';
import type { InventoryData } from './dashboard';
import * as XLSX from 'xlsx';

export function CsvUploader() {
  const { setInventoryData, setCsvString, setFileName } = useInventory();
  const { toast } = useToast();

  const handleFile = (file: File) => {
    if (!file) return;
    setFileName(file.name);
    const reader = new FileReader();

    if (
      file.type === 'text/csv' ||
      file.type === 'application/vnd.ms-excel' ||
      file.type ===
        'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
    ) {
      reader.onload = (e) => {
        const buffer = e.target?.result;
        if (buffer) {
          const workbook = XLSX.read(buffer, { type: 'array' });
          const sheetName = workbook.SheetNames[0];
          const worksheet = workbook.Sheets[sheetName];
          const json = XLSX.utils.sheet_to_json(worksheet);
          const csv = XLSX.utils.sheet_to_csv(worksheet);
          setCsvString(csv);
          processData(json as any[]);
        }
      };
      reader.readAsArrayBuffer(file);
    } else {
      toast({
        variant: 'destructive',
        title: 'Invalid File Type',
        description: 'Please upload a valid .csv or .xlsx file.',
      });
    }
  };

  const processData = (data: any[]) => {
    try {
      const numericHeaders = [
        'Opening Stock (Units)',
        'Purchase (Units)',
        'Sales (Units)',
        'Closing Stock (Units)',
        'Cost Price in INR',
        'Selling Price in INR',
        'Reorder Level',
      ];

      const processedData: InventoryData[] = data.map((row) => {
        const newRow: InventoryData = {};
        for (const key in row) {
          const trimmedKey = key.trim();
          if (numericHeaders.includes(trimmedKey)) {
            const numValue = parseFloat(String(row[key]));
            newRow[trimmedKey] = isNaN(numValue) ? 0 : numValue;
          } else {
            newRow[trimmedKey] = row[key];
          }
        }
        return newRow;
      });

      setInventoryData(processedData);
      toast({
        title: 'Success',
        description: 'Inventory data uploaded and analyzed successfully.',
      });
    } catch (error) {
      toast({
        variant: 'destructive',
        title: 'File Processing Error',
        description: 'Failed to process the file. Please check the format.',
      });
      console.error(error);
    }
  };

  const onDragOver = (e: React.DragEvent<HTMLDivElement>) => {
    e.preventDefault();
  };

  const onDrop = (e: React.DragEvent<HTMLDivElement>) => {
    e.preventDefault();
    const file = e.dataTransfer.files[0];
    handleFile(file);
  };

  return (
    <div
      className="flex flex-1 items-center justify-center rounded-lg border border-dashed shadow-sm p-8"
      onDragOver={onDragOver}
      onDrop={onDrop}
    >
      <div className="flex flex-col items-center gap-2 text-center">
        <Upload className="h-12 w-12 text-muted-foreground" />
        <h3 className="font-heading text-2xl font-bold tracking-tight">
          Upload Your Inventory Sheet
        </h3>
        <p className="text-sm text-muted-foreground">
          Drag and drop a .csv or .xlsx file here or click to select a file.
        </p>
        <label
          htmlFor="file-upload"
          className="mt-4 cursor-pointer inline-flex items-center justify-center rounded-md text-sm font-medium ring-offset-background transition-colors focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-ring focus-visible:ring-offset-2 disabled:pointer-events-none disabled:opacity-50 bg-primary text-primary-foreground hover:bg-primary/90 h-10 px-4 py-2"
        >
          Select File
        </label>
        <input
          id="file-upload"
          type="file"
          className="hidden"
          accept=".csv, application/vnd.openxmlformats-officedocument.spreadsheetml.sheet, application/vnd.ms-excel"
          onChange={(e) => e.target.files && handleFile(e.target.files[0])}
        />
      </div>
    </div>
  );
}
